<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Content;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class ContentController extends Controller
{
    public function home()
    {
        $contents = Content::whereIn('key', [
            'home_hero_title',
            'home_hero_subtitle',
            'home_description',
            'home_services_title',
            'home_services_description'
        ])->get();

        return view('admin.contents.home', compact('contents'));
    }

    public function about()
    {
        $contents = Content::whereIn('key', [
            'about_profile',
            'about_vision',
            'about_mission',
            'about_founder_image'
        ])->get();

        return view('admin.contents.about', compact('contents'));
    }

    public function updateHome(Request $request)
    {
        $validated = $request->validate([
            'contents' => 'array',
            'contents.*.key' => 'required|string',
            'contents.*.value' => 'required|string',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,svg,gif|max:2048'
        ]);

        foreach ($validated['contents'] as $content) {
            Content::updateOrCreate(
                ['key' => $content['key']],
                ['value' => $content['value']]
            );
        }

        // Handle logo upload
        if ($request->hasFile('logo')) {
            $file = $request->file('logo');
            $filename = 'logo-' . time() . '.' . $file->getClientOriginalExtension();
            
            // Ensure logo directory exists
            if (!Storage::disk('public')->exists('logo')) {
                Storage::disk('public')->makeDirectory('logo', 0755, true);
            }
            
            // Delete old logo if exists
            $oldLogo = Content::where('key', 'logo')->value('logo');
            if ($oldLogo && Storage::disk('public')->exists($oldLogo)) {
                Storage::disk('public')->delete($oldLogo);
            }
            
            // Store new logo
            $path = $file->storeAs('logo', $filename, 'public');
            
            // Update database
            Content::updateOrCreate(
                ['key' => 'logo'],
                ['value' => '', 'logo' => 'logo/' . $filename]
            );
        }

        return redirect()->back()->with('success', 'Home content updated successfully!');
    }

    public function updateAbout(Request $request)
    {
        $validated = $request->validate([
            'contents' => 'array',
            'contents.*.key' => 'required|string',
            'contents.*.value' => 'required|string',
            'founder_image' => 'nullable|image|mimes:jpeg,png,jpg,svg,gif|max:10240' // 10MB
        ]);

        foreach ($validated['contents'] as $content) {
            Content::updateOrCreate(
                ['key' => $content['key']],
                ['value' => $content['value']]
            );
        }

        // ✅ FIX: SIMPAN FOTO KE LOKASI YANG BENAR
        if ($request->hasFile('founder_image')) {
            $image = $request->file('founder_image');
            $filename = 'founder-' . time() . '.' . $image->getClientOriginalExtension();
            
            // ✅ FIX: HAPUS 'public/' DARI PATH
            // ❌ SALAH: $path = $image->storeAs('public/founders', $filename, 'public');
            // ✅ BENAR:
            $path = $image->storeAs('founders', $filename, 'public');
            
            // ✅ Hapus file lama
            $oldImage = Content::where('key', 'about_founder_image')->value('value');
            if ($oldImage && Storage::disk('public')->exists('founders/' . $oldImage)) {
                Storage::disk('public')->delete('founders/' . $oldImage);
            }
            
            // ✅ Simpan nama file ke database
            Content::updateOrCreate(
                ['key' => 'about_founder_image'],
                ['value' => $filename]
            );
        }

        return redirect()->back()->with('success', 'About content updated successfully!');
    }
}