<?php

namespace App\Http\Controllers;

use App\Models\Content;
use App\Models\CVOrder;
use Illuminate\Http\Request;

class CVOrderController extends Controller
{
    public function create()
    {
        return view('orders.create');
    }

    public function store(Request $request)
    {
        // Validasi data utama (hanya field wajib)
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'phone' => 'required|string|max:20',
            'job_position' => 'required|string|max:255',
            'education' => 'required|string',
            'education_from' => 'required|string',
            'education_to' => 'required|string',
            'hard_skills' => 'required|string',
            'soft_skills' => 'required|string',
            'cv_design' => 'required|numeric|between:1,10',
            
            // ✅ TAMBAHKAN ABOUT ME (opsional)
            'about_me' => 'nullable|string',
            
            // Work Experience (SEMUA OPSIONAL - untuk fresh graduate)
            'work_experiences' => 'nullable|array',
            'work_experiences.*.job_title' => 'nullable|string',
            'work_experiences.*.company' => 'nullable|string',
            'work_experiences.*.start_date' => 'nullable|string',
            'work_experiences.*.end_date' => 'nullable|string',
            'work_experiences.*.description' => 'nullable|string',
            
            // Organizational Experience (opsional)
            'organizational_experiences' => 'nullable|array',
            'organizational_experiences.*.position' => 'nullable|string',
            'organizational_experiences.*.organization' => 'nullable|string',
            'organizational_experiences.*.start_date' => 'nullable|string',
            'organizational_experiences.*.end_date' => 'nullable|string',
            'organizational_experiences.*.description' => 'nullable|string',
            
            // Certifications (opsional)
            'certifications' => 'nullable|array',
            'certifications.*.name' => 'nullable|string',
            'certifications.*.issue_date' => 'nullable|string',
            'certifications.*.expiry_date' => 'nullable|string',
            'certifications.*.description' => 'nullable|string',
            
            // Languages (opsional)
            'languages' => 'nullable|array',
            'languages.*' => 'nullable|array',
            'languages.*.name' => 'nullable|string',
            'languages.*.proficiency' => 'nullable|string',
            
        ]);

        /**
         * PENTING: Filter data yang akan disimpan ke database
         * Hanya field yang ADA di tabel cv_orders yang akan di-insert
         */
        $allowedFields = (new CVOrder())->getFillable();
        
        // Data untuk disimpan ke database (hanya field yang ada di tabel)
        $dataForDatabase = [];
        foreach ($allowedFields as $field) {
            if (isset($validated[$field])) {
                $dataForDatabase[$field] = $validated[$field];
            }
        }
        
        // Set default status jika tidak ada
        if (!isset($dataForDatabase['status'])) {
            $dataForDatabase['status'] = 'pending';
        }

        // Simpan pesanan CV ke DATABASE
        $order = CVOrder::create($dataForDatabase);

        // Ambil desain yang dipilih
        $selectedId = $validated['cv_design'];
        $title = Content::getValue('cv_catalog_' . $selectedId . '_title', 'CV Design ' . $selectedId);
        $price = Content::getValue('cv_catalog_' . $selectedId . '_price', 'Rp 0');

        // Ambil nomor WhatsApp dari database
        $whatsappNumber = Content::getValue('whatsapp_number', '6288801026800');

        // Format pesan WhatsApp (gunakan data dari $validated, BUKAN dari $order)
        $message = "Halo RG Digital Solution, saya ingin memesan CV dengan detail:\n\n" .
                   "Nama: {$validated['name']}\n" .
                   "Email: {$validated['email']}\n" .
                   "Telepon: {$validated['phone']}\n" .
                   "Posisi: {$validated['job_position']}\n" .
                   "Pendidikan: {$validated['education']} ({$validated['education_from']} - {$validated['education_to']})\n";

        // ✅ TAMBAHKAN DESKRIPSI DIRI (about_me)
        if (!empty($validated['about_me'])) {
            $message .= "\nTentang Diri:\n" . $validated['about_me'] . "\n";
        }

        // Tambahkan Work Experience (hanya jika ada data yang valid)
        if (!empty($validated['work_experiences'])) {
            $hasValidWorkExp = false;
            foreach ($validated['work_experiences'] as $exp) {
                if (!empty($exp['job_title']) && !empty($exp['company'])) {
                    $hasValidWorkExp = true;
                    break;
                }
            }
            
            if ($hasValidWorkExp) {
                $message .= "\nPengalaman Kerja:\n";
                $index = 1;
                foreach ($validated['work_experiences'] as $exp) {
                    if (!empty($exp['job_title']) && !empty($exp['company'])) {
                        $message .= $index . ". {$exp['job_title']} di {$exp['company']} ({$exp['start_date']} - {$exp['end_date']})";
                        if (!empty($exp['description'])) {
                            $message .= "\n   - {$exp['description']}";
                        }
                        $message .= "\n";
                        $index++;
                    }
                }
            }
        }

        // Tambahkan Organizational Experience (hanya jika ada data yang valid)
        if (!empty($validated['organizational_experiences'])) {
            $hasValidOrgExp = false;
            foreach ($validated['organizational_experiences'] as $org) {
                if (!empty($org['position']) && !empty($org['organization'])) {
                    $hasValidOrgExp = true;
                    break;
                }
            }
            
            if ($hasValidOrgExp) {
                $message .= "\nPengalaman Organisasi:\n";
                $index = 1;
                foreach ($validated['organizational_experiences'] as $org) {
                    if (!empty($org['position']) && !empty($org['organization'])) {
                        $message .= $index . ". {$org['position']} di {$org['organization']} ({$org['start_date']} - {$org['end_date']})";
                        if (!empty($org['description'])) {
                            $message .= "\n   - {$org['description']}";
                        }
                        $message .= "\n";
                        $index++;
                    }
                }
            }
        }

        // Tambahkan Hard Skills & Soft Skills
        $message .= "\nHard Skills:\n- " . str_replace("\n", "\n- ", trim($validated['hard_skills'])) . "\n";
        $message .= "\nSoft Skills:\n- " . str_replace("\n", "\n- ", trim($validated['soft_skills'])) . "\n";

        // Tambahkan Certifications (hanya jika ada data yang valid)
        if (!empty($validated['certifications'])) {
            $hasValidCert = false;
            foreach ($validated['certifications'] as $cert) {
                if (!empty($cert['name'])) {
                    $hasValidCert = true;
                    break;
                }
            }
            
            if ($hasValidCert) {
                $message .= "\nSertifikasi:\n";
                $index = 1;
                foreach ($validated['certifications'] as $cert) {
                    if (!empty($cert['name'])) {
                        $message .= $index . ". {$cert['name']}";
                        if (!empty($cert['issue_date'])) {
                            $message .= " (Diterbitkan: {$cert['issue_date']})";
                        }
                        if (!empty($cert['expiry_date'])) {
                            $message .= " (Kadaluarsa: {$cert['expiry_date']})";
                        }
                        if (!empty($cert['description'])) {
                            $message .= "\n   - {$cert['description']}";
                        }
                        $message .= "\n";
                        $index++;
                    }
                }
            }
        }

        // Tambahkan Languages (hanya jika ada data yang valid)
        if (!empty($validated['languages'])) {
            $hasValidLang = false;
            foreach ($validated['languages'] as $lang) {
                if (!empty($lang['name']) && !empty($lang['proficiency'])) {
                    $hasValidLang = true;
                    break;
                }
            }
            
            if ($hasValidLang) {
                $message .= "\nBahasa:\n";
                foreach ($validated['languages'] as $lang) {
                    if (!empty($lang['name']) && !empty($lang['proficiency'])) {
                        $message .= "- {$lang['name']} ({$lang['proficiency']})\n";
                    }
                }
            }
        }

        // Tambahkan desain CV yang dipilih
        $message .= "\nDesain CV yang Dipilih:\n" .
                   "- {$title} - {$price}";

        // HAPUS SPASI BERLEBIH DI URL WHATSAPP
        $whatsappNumber = preg_replace('/\s+/', '', $whatsappNumber);
        $whatsappUrl = "https://wa.me/{$whatsappNumber}?text=" . urlencode($message);

        return redirect()->away($whatsappUrl);
    }
}